<?php
/**
 * Auth Sınıfı - Kimlik Doğrulama
 * Session bazlı authentication
 */

class Auth {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();

        // Session başlatılmamışsa başlat
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    /**
     * Kullanıcı girişi
     */
    public function login($username, $password) {
        $sql = "SELECT * FROM users WHERE username = ? AND is_active = 1";
        $user = $this->db->selectOne($sql, [$username]);

        if ($user && password_verify($password, $user['password_hash'])) {
            // Session'a kullanıcı bilgilerini kaydet
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['logged_in'] = true;

            // Aktivite log
            $this->logActivity($user['id'], 'login', 'users', $user['id']);

            return true;
        }

        return false;
    }

    /**
     * Kullanıcı çıkışı
     */
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $this->logActivity($_SESSION['user_id'], 'logout', 'users', $_SESSION['user_id']);
        }

        // Session'ı temizle
        $_SESSION = [];
        session_destroy();

        return true;
    }

    /**
     * Kullanıcı giriş yapmış mı?
     */
    public function isLoggedIn() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }

    /**
     * Mevcut kullanıcı bilgisi
     */
    public function user() {
        if (!$this->isLoggedIn()) {
            return null;
        }

        return [
            'id' => $_SESSION['user_id'] ?? null,
            'username' => $_SESSION['username'] ?? null,
            'full_name' => $_SESSION['full_name'] ?? null,
            'role' => $_SESSION['role'] ?? null,
        ];
    }

    /**
     * Kullanıcı ID'si
     */
    public function userId() {
        return $_SESSION['user_id'] ?? null;
    }

    /**
     * Yetki kontrolü
     */
    public function hasRole($role) {
        if (!$this->isLoggedIn()) {
            return false;
        }

        $userRole = $_SESSION['role'] ?? null;

        // Admin her şeye erişebilir
        if ($userRole === 'admin') {
            return true;
        }

        // Belirtilen role sahip mi?
        if (is_array($role)) {
            return in_array($userRole, $role);
        }

        return $userRole === $role;
    }

    /**
     * Login sayfasına yönlendir
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: ' . BASE_URL . '/login');
            exit;
        }
    }

    /**
     * Yetki kontrolü ile yönlendirme
     */
    public function requireRole($role) {
        $this->requireLogin();

        if (!$this->hasRole($role)) {
            header('Location: ' . BASE_URL . '/dashboard');
            exit;
        }
    }

    /**
     * CSRF Token oluştur
     */
    public function generateCsrfToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    /**
     * CSRF Token doğrula
     */
    public function verifyCsrfToken($token) {
        if (!isset($_SESSION['csrf_token'])) {
            return false;
        }
        return hash_equals($_SESSION['csrf_token'], $token);
    }

    /**
     * Aktivite log
     */
    private function logActivity($userId, $action, $tableName = null, $recordId = null) {
        try {
            $data = [
                'user_id' => $userId,
                'action' => $action,
                'table_name' => $tableName,
                'record_id' => $recordId,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
            ];

            $this->db->insert('activity_logs', $data);
        } catch (Exception $e) {
            // Log hatası sessizce göz ardı edilir
            error_log("Activity log error: " . $e->getMessage());
        }
    }

    /**
     * Yeni kullanıcı oluştur
     */
    public function register($username, $email, $password, $fullName, $role = 'viewer') {
        // Kullanıcı var mı kontrol et
        $existing = $this->db->selectOne("SELECT id FROM users WHERE username = ? OR email = ?", [$username, $email]);

        if ($existing) {
            return false;
        }

        $data = [
            'username' => $username,
            'email' => $email,
            'password_hash' => password_hash($password, PASSWORD_BCRYPT),
            'full_name' => $fullName,
            'role' => $role,
            'is_active' => 1
        ];

        return $this->db->insert('users', $data);
    }
}
