<?php
/**
 * CheckController - Birleşik Çek Takibi (Alınan + Verilen)
 */
class CheckController
{
    private $db;
    private $auth;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    public function index()
    {
        $this->ensureTablesExist();

        // Verilen Çekler (Supplier Checks)
        $verilenCekler = $this->db->select("
            SELECT
                sc.id,
                sc.supplier_id,
                sc.supplier_transaction_id,
                sc.check_number,
                sc.bank_name,
                sc.amount,
                sc.issue_date,
                sc.due_date,
                sc.status,
                sc.description,
                sc.created_at,
                s.supplier_name,
                st.reference_number
            FROM supplier_checks sc
            INNER JOIN suppliers s ON s.id = sc.supplier_id
            INNER JOIN supplier_transactions st ON st.id = sc.supplier_transaction_id
            ORDER BY sc.due_date ASC, sc.created_at DESC
        ");

        // Alınan Çekler (Received Checks)
        $alinanCekler = $this->db->select("
            SELECT
                rc.id,
                rc.customer_name,
                rc.bank_name,
                rc.amount,
                rc.due_date,
                rc.check_number,
                rc.status,
                rc.description,
                rc.created_at
            FROM received_checks rc
            ORDER BY rc.due_date ASC, rc.created_at DESC
        ");

        // Verilen Çekler Özeti
        $verilenSummary = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN amount ELSE 0 END) as pending_total,
                SUM(CASE WHEN status = 'TAHSIL_EDILDI' THEN amount ELSE 0 END) as collected_total,
                SUM(amount) as grand_total
            FROM supplier_checks
        ");

        // Alınan Çekler Özeti
        $alinanSummary = $this->db->selectOne("
            SELECT
                COUNT(*) as total_count,
                SUM(CASE WHEN status = 'BEKLEMEDE' THEN amount ELSE 0 END) as pending_total,
                SUM(CASE WHEN status = 'TAHSIL_EDILDI' THEN amount ELSE 0 END) as collected_total,
                SUM(CASE WHEN status = 'CIRO_EDILDI' THEN amount ELSE 0 END) as endorsed_total,
                SUM(amount) as grand_total
            FROM received_checks
        ");

        $auth = $this->auth;
        $pageTitle = 'Çek Takibi';
        $currentPage = 'checks';

        require_once __DIR__ . '/../views/checks/index.php';
    }

    // Verilen Çek Aksiyonları
    public function markVerilenCollected($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $updated = $this->db->update('supplier_checks', [
            'status' => 'TAHSIL_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Verilen çek tahsil edildi olarak işaretlendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/checks');
    }

    public function markVerilenPending($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $updated = $this->db->update('supplier_checks', [
            'status' => 'BEKLEMEDE'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Verilen çek beklemede olarak güncellendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/checks');
    }

    // Alınan Çek Aksiyonları
    public function markAlinanCollected($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'TAHSIL_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Alınan çek tahsil edildi olarak işaretlendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/checks');
    }

    public function markAlinanPending($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'BEKLEMEDE'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Alınan çek beklemede olarak güncellendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/checks');
    }

    public function markAlinanEndorsed($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $updated = $this->db->update('received_checks', [
            'status' => 'CIRO_EDILDI'
        ], 'id = ?', [$id]);

        if ($updated !== false) {
            Helper::setFlash('success', 'Alınan çek ciro edildi olarak işaretlendi.');
        } else {
            Helper::setFlash('danger', 'İşlem başarısız oldu.');
        }

        Helper::redirect('/checks');
    }

    public function deleteAlinan($id)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/checks');
        }

        $result = $this->db->delete('received_checks', 'id = ?', [$id]);

        if ($result) {
            Helper::setFlash('success', 'Alınan çek kaydı başarıyla silindi.');
        } else {
            Helper::setFlash('danger', 'Çek kaydı silinemedi.');
        }

        Helper::redirect('/checks');
    }

    private function ensureTablesExist()
    {
        // Ensure supplier_checks table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS supplier_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                supplier_id INT NOT NULL,
                supplier_transaction_id INT NOT NULL,
                check_number VARCHAR(100) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                issue_date DATE NOT NULL,
                due_date DATE NOT NULL,
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_supplier_transaction (supplier_transaction_id),
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_supplier (supplier_id),
                FOREIGN KEY (supplier_id) REFERENCES suppliers(id) ON DELETE CASCADE,
                FOREIGN KEY (supplier_transaction_id) REFERENCES supplier_transactions(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");

        // Ensure received_checks table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS received_checks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                customer_name VARCHAR(200) NOT NULL,
                bank_name VARCHAR(100),
                amount DECIMAL(15,2) NOT NULL,
                due_date DATE NOT NULL,
                check_number VARCHAR(100),
                status ENUM('BEKLEMEDE', 'TAHSIL_EDILDI', 'IADE', 'CIRO_EDILDI') DEFAULT 'BEKLEMEDE',
                description TEXT,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_due_date (due_date),
                INDEX idx_status (status),
                INDEX idx_customer_name (customer_name),
                FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_turkish_ci
        ");
    }
}
