<?php
/**
 * BankController - Banka Yönetimi
 */

class BankController {
    private $db;
    private $auth;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }

    public function index() {
        // BCH Summary
        $bchSummary = $this->db->selectOne("
            SELECT
                COUNT(*) as pending_count,
                SUM(amount) as pending_total
            FROM bank_bch
            WHERE status = 'BEKLEMEDE'
        ");

        // KMH Summary
        $kmhSummary = $this->db->selectOne("
            SELECT
                COUNT(*) as account_count,
                SUM(limit_amount) as limit_total,
                SUM(used_amount) as used_total,
                SUM(available_amount) as available_total
            FROM banks
            WHERE account_type = 'kmh'
        ");

        // Installment Loans Summary
        $installmentSummary = $this->db->selectOne("
            SELECT
                COUNT(*) as loan_count,
                SUM(monthly_payment) as monthly_total,
                SUM(remaining_installments) as total_remaining
            FROM bank_installment_loans
            WHERE status <> 'TAMAMLANDI'
        ");

        $auth = $this->auth;
        $pageTitle = 'Banka Borç Takibi';
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/index.php';
    }

    public function view($id) {
        $bank = $this->db->selectOne("SELECT * FROM banks WHERE id = ?", [$id]);

        if (!$bank) {
            Helper::setFlash('Banka bulunamadı!', 'danger');
            Helper::redirect('/banks');
        }

        $balance = $this->calculateBalance($id);

        $transactions = $this->db->select("
            SELECT * FROM bank_transactions
            WHERE bank_id = ?
            ORDER BY transaction_date DESC, created_at DESC
        ", [$id]);

        $auth = $this->auth;
        $pageTitle = $bank['bank_name'];
        $currentPage = 'banks';

        require_once __DIR__ . '/../views/banks/view.php';
    }

    public function saveTransaction($id) {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            Helper::redirect('/banks/view/' . $id);
        }

        $bank = $this->db->selectOne("SELECT * FROM banks WHERE id = ?", [$id]);
        if (!$bank) {
            Helper::setFlash('Banka bulunamadı!', 'danger');
            Helper::redirect('/banks');
        }

        $data = [
            'bank_id' => $id,
            'account_category' => $_POST['account_category'] ?? null,
            'transaction_type' => $_POST['transaction_type'] ?? '',
            'amount' => floatval($_POST['amount'] ?? 0),
            'transaction_date' => $_POST['transaction_date'] ?? date('Y-m-d'),
            'reference_number' => $_POST['reference_number'] ?? null,
            'description' => $_POST['description'] ?? null,
            'created_by' => $_SESSION['user_id'] ?? null
        ];

        if (empty($data['transaction_type']) || $data['amount'] <= 0) {
            Helper::setFlash('İşlem tipi ve tutar gerekli!', 'danger');
            Helper::redirect('/banks/view/' . $id);
        }

        $result = $this->db->insert('bank_transactions', $data);

        if ($result) {
            Helper::setFlash('İşlem başarıyla kaydedildi!', 'success');
        } else {
            Helper::setFlash('İşlem kaydedilemedi!', 'danger');
        }

        Helper::redirect('/banks/view/' . $id);
    }

    public function deleteTransaction($bankId, $transactionId) {
        $bank = $this->db->selectOne("SELECT * FROM banks WHERE id = ?", [$bankId]);
        if (!$bank) {
            Helper::setFlash('Banka bulunamadı!', 'danger');
            Helper::redirect('/banks');
        }

        $transaction = $this->db->selectOne("
            SELECT * FROM bank_transactions WHERE id = ? AND bank_id = ?
        ", [$transactionId, $bankId]);

        if (!$transaction) {
            Helper::setFlash('İşlem bulunamadı!', 'danger');
            Helper::redirect('/banks/view/' . $bankId);
        }

        $result = $this->db->delete('bank_transactions', ['id' => $transactionId]);

        if ($result) {
            Helper::setFlash('İşlem başarıyla silindi!', 'success');
        } else {
            Helper::setFlash('İşlem silinemedi!', 'danger');
        }

        Helper::redirect('/banks/view/' . $bankId);
    }

    private function calculateBalance($bankId) {
        $result = $this->db->selectOne("
            SELECT
                SUM(CASE
                    WHEN transaction_type = 'borrow' THEN amount
                    WHEN transaction_type = 'payment' THEN -amount
                    WHEN transaction_type = 'adjustment' THEN amount
                    ELSE 0
                END) as balance,
                SUM(CASE WHEN transaction_type = 'borrow' THEN amount ELSE 0 END) as total_borrows,
                SUM(CASE WHEN transaction_type = 'payment' THEN amount ELSE 0 END) as total_payments,
                COUNT(*) as transaction_count
            FROM bank_transactions
            WHERE bank_id = ?
        ", [$bankId]);

        return [
            'balance' => $result['balance'] ?? 0,
            'total_borrows' => $result['total_borrows'] ?? 0,
            'total_payments' => $result['total_payments'] ?? 0,
            'transaction_count' => $result['transaction_count'] ?? 0
        ];
    }
}
